<?php
// db/clients-update.php

// Configuración de errores para depuración
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();
header('Content-Type: application/json');

include_once 'db_connection.php'; 
include_once 'client-services-sync.php'; 

$response = ['success' => false, 'message' => ''];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Error de JSON en clients-update.php: " . json_last_error_msg());
        $response['message'] = 'Error al decodificar datos de solicitud.';
        echo json_encode($response);
        exit();
    }
    if (empty($input)) {
        error_log("Datos de entrada vacíos o JSON inválido en clients-update.php.");
        $response['message'] = 'Datos de solicitud vacíos o no válidos.';
        echo json_encode($response);
        exit();
    }

    $client_id = $input['id'] ?? null;
    $first_name = trim($input['first_name'] ?? '');
    $last_name = trim($input['last_name'] ?? '');
    $email = trim($input['email'] ?? '');
    
    $phone = !empty(trim($input['phone'] ?? '')) ? trim($input['phone']) : null;
    $mobile = !empty(trim($input['mobile'] ?? '')) ? trim($input['mobile']) : null;
    $street_address = !empty(trim($input['street_address'] ?? '')) ? trim($input['street_address']) : null;
    $city = !empty(trim($input['city'] ?? '')) ? trim($input['city']) : null;
    $state_province = !empty(trim($input['state_province'] ?? '')) ? trim($input['state_province']) : null;
    $zip_code = !empty(trim($input['zip_code'] ?? '')) ? trim($input['zip_code']) : null;

    $assigned_service_ids = is_array($input['assigned_service_ids'] ?? null) ? $input['assigned_service_ids'] : []; 

    // ✅ CORRECCIÓN 1: Convertir el valor booleano a un entero (1 o 0)
    $is_recurring_int = isset($input['is_recurring']) && $input['is_recurring'] ? 1 : 0;

    // Validar datos básicos
    if (empty($client_id) || empty($first_name)) {
        $response['message'] = 'ID de cliente y nombre son obligatorios.';
        echo json_encode($response);
        exit();
    }
    if (empty($email)) {
        $response['message'] = 'El email del cliente es obligatorio.';
        echo json_encode($response);
        exit();
    }
    if (empty($phone) && empty($mobile)) {
        $response['message'] = 'Se requiere al menos un número de teléfono (fijo o móvil).';
        echo json_encode($response);
        exit();
    }

    try {
        $pdo->beginTransaction();

        // 1. Actualizar la información del cliente
        $stmt = $pdo->prepare("UPDATE clients SET first_name = ?, last_name = ?, email = ?, phone = ?, mobile = ?, street_address = ?, city = ?, state_province = ?, zip_code = ?, is_recurring = ? WHERE id = ?");
        
        // ✅ CORRECCIÓN 2: Usar la nueva variable entera en la consulta
        $stmt->execute([$first_name, $last_name, $email, $phone, $mobile, $street_address, $city, $state_province, $zip_code, $is_recurring_int, $client_id]);

        // 2. Sincronizar los servicios asignados
        if (!syncClientServices($pdo, (int)$client_id, $assigned_service_ids)) {
            throw new Exception("Error al sincronizar servicios.");
        }

        $pdo->commit();
        $response['success'] = true;
        $response['message'] = 'Cambios guardados con éxito.';
        
        // Devolver el objeto cliente completo
        $response['client'] = [
            'id' => (int)$client_id,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $email,
            'phone' => $phone,
            'mobile' => $mobile,
            'street_address' => $street_address,
            'city' => $city,
            'state_province' => $state_province,
            'zip_code' => $zip_code,
            'assigned_service_ids' => $assigned_service_ids,
            'is_recurring' => (bool)$is_recurring_int // Devolvemos un booleano a JS, lo cual es correcto
        ];

    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error al actualizar cliente (PDO): " . $e->getMessage());
        $response['message'] = 'Error de base de datos al actualizar cliente: ' . $e->getMessage();
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error general al actualizar cliente: " . $e->getMessage());
        $response['message'] = 'Error interno al actualizar cliente: ' . $e->getMessage();
    }
} else {
    error_log("clients-update.php: Solicitud recibida no es POST. Método: " . $_SERVER['REQUEST_METHOD']);
    $response['message'] = 'Método de solicitud no válido.';
}

echo json_encode($response);
exit();
?>