<?php
// db/generate-recurring-instances.php
// Este script debe ejecutarse una vez al día (idealmente con un Cron Job).
include_once 'db_connection.php';

try {
    // 1. Obtener todas las reglas de finanzas recurrentes activas
    $currentDate = new DateTime();
    $currentYearMonth = $currentDate->format('Y-m');
    $currentDay = $currentDate->format('j');

    // Selecciona reglas que:
    // - Están activas (start_date es hoy o en el pasado).
    // - No han expirado (end_date es nulo o en el futuro).
    $sql = "SELECT * FROM recurring_finances 
            WHERE start_date <= CURDATE() 
            AND (end_date IS NULL OR end_date >= CURDATE())";
    
    $stmt = $pdo->query($sql);
    $recurringRules = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // echo "Iniciando generador de instancias recurrentes... \n"; // Comentado

    foreach ($recurringRules as $rule) {
        $lastGeneratedDate = !empty($rule['last_generated_date']) ? new DateTime($rule['last_generated_date']) : null;
        $lastGeneratedYearMonth = $lastGeneratedDate ? $lastGeneratedDate->format('Y-m') : null;

        // 2. Verificar si ya se generó una transacción para esta regla en el mes actual
        if ($lastGeneratedYearMonth === $currentYearMonth) {
            // echo "Regla #{$rule['id']} ('{$rule['description']}') ya generada para {$currentYearMonth}. Omitiendo.\n"; // Comentado
            continue; // Ya se generó para este mes, pasar a la siguiente regla
        }

        // 3. Verificar si el día de recurrencia ya pasó en el mes actual
        if ($currentDay >= $rule['recurrence_day_of_month']) {
            
            // Construir la fecha de la transacción para el mes actual
            $transactionDateStr = $currentYearMonth . '-' . str_pad($rule['recurrence_day_of_month'], 2, '0', STR_PAD_LEFT);
            $transactionDate = new DateTime($transactionDateStr);

            // Doble chequeo: solo generar si la fecha de la transacción está dentro del rango de la regla.
            $startDate = new DateTime($rule['start_date']);
            $endDate = !empty($rule['end_date']) ? new DateTime($rule['end_date']) : null;

            if ($transactionDate >= $startDate && (!$endDate || $transactionDate <= $endDate)) {
                
                // 4. Insertar la nueva transacción en la tabla `finances`
                // echo "Generando instancia para la regla #{$rule['id']} ('{$rule['description']}') para la fecha {$transactionDateStr}... "; // Comentado
                
                $insertSql = "INSERT INTO finances (type, description, amount, transaction_date, client_id) VALUES (?, ?, ?, ?, ?)";
                $insertStmt = $pdo->prepare($insertSql);
                $success = $insertStmt->execute([
                    $rule['type'],
                    $rule['description'] . " (Recurrente)", // Añadir un identificador
                    $rule['amount'],
                    $transactionDateStr,
                    $rule['client_id']
                ]);

                if ($success) {
                    // 5. Actualizar la fecha de última generación en la tabla `recurring_finances`
                    $updateSql = "UPDATE recurring_finances SET last_generated_date = CURDATE() WHERE id = ?";
                    $updateStmt = $pdo->prepare($updateSql);
                    $updateStmt->execute([$rule['id']]);
                    // echo "¡Éxito!\n"; // Comentado
                } else {
                    // echo "¡Falló la inserción!\n"; // Comentado
                    error_log("Fallo al insertar la transacción recurrente para la regla ID: {$rule['id']}");
                }
            }
        } else {
            // echo "Regla #{$rule['id']} ('{$rule['description']}') - día de recurrencia ({$rule['recurrence_day_of_month']}) aún no ha llegado este mes. Omitiendo.\n"; // Comentado
        }
    }

    // echo "Proceso finalizado.\n"; // Comentado

} catch (PDOException $e) {
    error_log("PDO Error en generate-recurring-instances.php: " . $e->getMessage());
    // die("Error de Base de Datos: " . $e->getMessage()); // Descomentar solo para depuración directa
} catch (Exception $e) {
    error_log("General Error en generate-recurring-instances.php: " . $e->getMessage());
    // die("Error General: " . $e->getMessage()); // Descomentar solo para depuración directa
}
?>