<?php
// db/invoice-create.php
ob_start();
error_reporting(0);
ini_set('display_errors', 0);

include_once 'db_connection.php';

$response = [];

try {
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Error: JSON inválido recibido. ' . json_last_error_msg());
    }

    // Validar datos requeridos para la factura principal (ahora solo client_id)
    if (empty($data['invoice_date']) || empty($data['status']) || empty($data['client_id']) || !isset($data['items'])) {
        throw new Exception('Datos incompletos para la factura. Fecha, estado, ID de cliente e ítems son requeridos.');
    }
    if (!is_array($data['items']) || empty($data['items'])) {
        throw new Exception('Los ítems de la factura deben ser un array no vacío.');
    }

    $client_id = filter_var($data['client_id'], FILTER_VALIDATE_INT);
    // Eliminar lead_id por completo, ya no es relevante para facturas
    $lead_id = null; // Siempre será null

    if ($client_id === false) {
        throw new Exception('ID de cliente inválido.');
    }

    $invoice_date = $data['invoice_date'];
    $due_date = !empty($data['due_date']) ? $data['due_date'] : null; // Campo opcional
    $status = trim($data['status']);
    
    // Forzar estado inicial a 'pendiente' si es nueva creación
    if (empty($data['id'])) { 
        $status = 'pendiente'; // Estado por defecto para nuevas facturas
    }

    $tax_rate = filter_var($data['tax_rate'] ?? 0.00, FILTER_VALIDATE_FLOAT);
    $items = $data['items']; // Array de [description, unit_price, quantity]

    // Calcular subtotal y total_amount antes de insertar
    $subtotal = 0;
    foreach ($items as $item) {
        if (!isset($item[0], $item[1], $item[2])) { // description, unit_price, quantity
            throw new Exception('Formato de ítem inválido. Cada ítem debe tener descripción, precio y cantidad.');
        }
        $item_price = filter_var($item[1], FILTER_VALIDATE_FLOAT);
        $item_quantity = filter_var($item[2], FILTER_VALIDATE_INT);
        if ($item_price === false || $item_quantity === false || $item_quantity <= 0) {
            throw new Exception('Precio o cantidad de ítem inválido.');
        }
        $subtotal += ($item_price * $item_quantity);
    }
    $total_amount = $subtotal + ($subtotal * ($tax_rate / 100));

    $pdo->beginTransaction();

    // Generar un external_id único para la factura
    $external_id = 'fac-' . uniqid();

    // Insertar la factura principal (eliminado lead_id de la inserción)
    $stmt = $pdo->prepare(
        "INSERT INTO invoices (external_id, client_id, lead_id, invoice_date, due_date, status, tax_rate, subtotal, total_amount)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
    );
    $stmt->execute([
        $external_id,
        $client_id,
        $lead_id, // Siempre será null
        $invoice_date,
        $due_date,
        $status,
        $tax_rate,
        $subtotal,
        $total_amount
    ]);

    $invoice_id = $pdo->lastInsertId();

    // Insertar los ítems de la factura
    $stmt_items = $pdo->prepare(
        "INSERT INTO invoice_items (invoice_id, description, unit_price, quantity)
         VALUES (?, ?, ?, ?)"
    );
    foreach ($items as $item) {
        $stmt_items->execute([
            $invoice_id,
            $item[0], // description
            filter_var($item[1], FILTER_VALIDATE_FLOAT), // unit_price
            filter_var($item[2], FILTER_VALIDATE_INT) // quantity
        ]);
    }

    $pdo->commit();

    $response = [
        'success' => true,
        'message' => 'Factura agregada con éxito.',
        'id' => $invoice_id,
        'external_id' => $external_id
    ];

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("PDO Error en invoice-create.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => 'Error de Base de Datos: ' . $e->getMessage()];
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("General Error en invoice-create.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => $e->getMessage()];
}

ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>