<?php
session_start();
include_once 'db/db_connection.php';
// El generador se debe ejecutar para que la tabla `finanzas` tenga las instancias de cada mes.
include_once 'db/generate-recurring-instances.php'; 

// 1. Obtener TODAS las transacciones que ya existen en la tabla `finanzas`.
$all_transactions = [];
try {
    $stmt = $pdo->query("
        SELECT 
            f.id, 
            f.type, 
            f.description, 
            f.amount, 
            f.transaction_date, 
            f.client_id,
            CONCAT(c.first_name, ' ', c.last_name) as client_name
        FROM finances f
        LEFT JOIN clients c ON f.client_id = c.id
        ORDER BY f.transaction_date DESC, f.id DESC
    ");
    $all_transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar finanzas desde la DB: " . $e->getMessage());
    // Error handling will be done in JS with showToast
}

// 2. Obtener TODAS las REGLAS de gastos recurrentes.
$all_recurring_expenses = [];
try {
    $stmt = $pdo->query("
        SELECT 
            rf.id, 
            rf.type, 
            rf.description, 
            rf.amount, 
            rf.recurrence_day_of_month,
            rf.start_date,
            rf.end_date,
            rf.client_id,
            CONCAT(c.first_name, ' ', c.last_name) as client_name
        FROM recurring_finances rf
        LEFT JOIN clients c ON rf.client_id = c.id
        WHERE rf.type = 'gasto'
        ORDER BY rf.recurrence_day_of_month ASC, rf.description ASC
    ");
    $all_recurring_expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar egresos recurrentes: " . $e->getMessage());
    // Error handling will be done in JS with showToast
}

// Convertir los arrays PHP a formato JSON para que JavaScript los pueda usar
$transactions_json = json_encode($all_transactions);
$recurring_expenses_json = json_encode($all_recurring_expenses);

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>LocalCRM ToolKit Dashboard | Finanzas | OrozDesign Multimedia</title>
    <meta name="description" content="Gestiona tus finanzas en LocalCRM: registra ingresos, gastos, facturas y obtén un resumen claro de tu flujo de caja">
    <meta name="robots" content="noindex, nofollow">
    <link rel="icon" type="image/png" href="img/favicon.png">
    <link rel="apple-touch-icon" href="img/apple-touch-icon.png">
    <?php include 'files/gtm-head.php'; ?>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Barlow:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <script src="https://unpkg.com/lucide@latest"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <link rel="stylesheet" href="style.css">
    <script src="files/header-manager.js"></script>
    <style>
        .autocomplete-suggestions{border:1px solid #ddd;background:#fff;max-height:150px;overflow-y:auto;position:absolute;z-index:100;width:100%;box-shadow:0 2px 5px rgba(0,0,0,0.1)}.autocomplete-suggestion-item{padding:8px 10px;cursor:pointer;border-bottom:1px solid #eee}.autocomplete-suggestion-item:last-child{border-bottom:none}.autocomplete-suggestion-item:hover{background-color:#f0f0f0}.styled-readonly-input{background-color:#e9e9ed;cursor:not-allowed;border:1px solid #d1d5db;color:#4b5563;font-weight:600;box-shadow:inset 0 1px 2px rgba(0,0,0,0.06)}.recurring-tag{background-color:#a78bfa;color:#4c1d95;padding:.25rem .5rem;border-radius:9999px;font-size:.75rem;font-weight:700;margin-left:.5rem;display:inline-flex;align-items:center}
        
        /* === NUEVAS REGLAS PARA TABLAS RESPONSIVAS EN FINANZAS === */
        @media (max-width: 767px) {
            /* Reglas base para todas las tablas que se quieren apilar */
            .responsive-table-stack tbody,
            .responsive-table-stack tr,
            .responsive-table-stack td {
                display: block;
                width: 100%;
            }

            .responsive-table-stack thead {
                display: none; /* Oculta el encabezado de la tabla original */
            }

            .responsive-table-stack tr {
                margin-bottom: 1rem; /* Espacio entre las "tarjetas" de fila */
                border: 1px solid #e2e8f0;
                border-radius: 0.5rem;
                padding: 0.5rem;
                background-color: white;
                box-shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
            }

            .responsive-table-stack td {
                text-align: right !important; /* Alinea el valor a la derecha */
                padding-left: 50% !important; /* Deja espacio para la etiqueta */
                position: relative;
                border-bottom: 1px solid #edf2f7; /* Separador entre campos */
                padding-top: 0.75rem;
                padding-bottom: 0.75rem;
                white-space: normal; /* Permite que el texto se ajuste */
                word-break: break-word; /* Rompe palabras largas */
            }

            .responsive-table-stack td:last-child {
                border-bottom: 0; /* No hay borde inferior en el último campo */
            }

            /* Etiqueta del campo (header) */
            .responsive-table-stack td::before {
                content: attr(data-label); /* Usa el atributo data-label como contenido */
                position: absolute;
                left: 0.75rem;
                width: calc(50% - 1.5rem); /* Ocupa la mitad izquierda */
                padding-right: 1rem;
                white-space: nowrap; /* Evita que la etiqueta se rompa */
                text-align: left;
                font-weight: bold;
                text-transform: uppercase;
                color: #4a5568;
                flex-shrink: 0;
            }

            /* Ajustes para celdas de acciones/botones */
            .responsive-table-stack td.actions-cell {
                text-align: center !important;
                padding-left: 0 !important;
                display: flex;
                justify-content: center;
                align-items: center;
                gap: 0.75rem; /* Espacio entre botones */
                min-height: 48px; /* Altura mínima para la fila de botones */
                flex-wrap: wrap; /* Permite que los botones se envuelvan si hay muchos */
            }

            .responsive-table-stack td.actions-cell::before {
                content: none; /* No mostrar etiqueta para celdas de acciones */
            }

            /* Asegurar que los botones dentro de celdas de acción se apilen si es necesario */
            .responsive-table-stack td.actions-cell button {
                width: auto; /* O auto para que no tomen todo el ancho */
                flex-grow: 1; /* Permite que crezcan para llenar el espacio, o 0 para que no */
                min-width: 80px; /* Asegura un tamaño mínimo para el botón */
            }

            /* Para la tabla de resumen histórico y gastos recurrentes */
            #finance-summary-table-body td,
            #recurring-expenses-table-body td {
                /* Asegúrate de que las reglas de data-label y apilamiento se apliquen */
                /* Ya se manejan por .responsive-table-stack */
            }
        }
        /* === FIN NUEVAS REGLAS === */
    </style>
</head>
<body data-page-title="CENTRO DE CONTROL FINANCIERO" 
      data-page-subtitle="ADMINISTRA TUS ENTRADAS Y SALIDAS ECONÓMICAS" 
      data-page-icon="wallet">

    <div id="toast-container" class="toast-container"></div>
        
<?php include 'files/gtm-body.php'; ?>

<div class="relative min-h-screen md:flex">
    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>
    <?php include 'menu.php'; ?>

    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
        </header>

        <div id="content-area" class="p-4 md:p-8 space-y-8">
            <section id="finanzas" class="dashboard-section">
                <div class="bg-white p-6 rounded-xl shadow-md">
                    <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-6 gap-4">
                        <div>
                            <h3 class="text-2xl font-extrabold text-gray-800 flex items-center gap-2">
                                <i data-lucide="trending-up" class="w-7 h-7 text-[var(--color-primary)]"></i> ANÁLISIS FINANCIERO
                            </h3>
                            <p class="text-gray-500 text-sm mt-1 uppercase">REVISA TUS INGRESOS, GASTOS Y EL FLUJO DE EFECTIVO.</p>
                        </div>
                        <div class="flex items-center gap-4 w-full md:w-auto">
                            <select id="finance-month-selector" class="w-full md:w-48 p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white"></select>
                            <div class="flex flex-row-reverse flex-wrap gap-2 w-full md:w-auto justify-end">
                                <button class="btn-secondary font-bold py-2 px-4 rounded-lg flex items-center uppercase flex-shrink-0" onclick="showFinancesPanel()">
                                    <i data-lucide="plus" class="w-5 h-5 mr-2"></i> ENTRADA
                                </button>
                                </div>
                        </div>
                    </div>
                    
<div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
    <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
        <i data-lucide="calendar-check" class="w-12 h-12 text-gray-700"></i>
        <div>
            <h3 class="text-lg font-extrabold text-gray-500 mb-1">INGRESOS TOTALES</h3>
            <p id="finance-total-income" class="text-5xl font-bold text-[var(--color-highlight)]">$0.00</p>
            <p id="finance-income-comparison" class="text-sm text-gray-400">&nbsp;</p>
        </div>
    </div>

    <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-secondary)]">
        <i data-lucide="trending-down" class="w-12 h-12 text-gray-700"></i>
        <div>
            <h3 class="text-lg font-extrabold text-gray-500 mb-1">GASTOS TOTALES</h3>
            <p id="finance-total-expense" class="text-5xl font-bold text-[var(--color-secondary)]">$0.00</p>
            <p id="finance-expense-comparison" class="text-sm text-gray-400">&nbsp;</p>
        </div>
    </div>

    <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-primary)]">
        <i data-lucide="wallet" class="w-12 h-12 text-gray-700"></i>
        <div>
            <h3 class="text-lg font-extrabold text-gray-500 mb-1">GANANCIA NETA</h3>
            <p id="finance-net-profit" class="text-5xl font-bold text-[var(--color-primary)]">$0.00</p>
            <p id="finance-profit-comparison" class="text-sm text-gray-400">&nbsp;</p>
        </div>
    </div>
</div>

                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        <div class="lg:col-span-1 bg-slate-50 rounded-xl p-6 flex flex-col items-center justify-center"><h4 class="text-lg font-extrabold text-gray-700 mb-4 uppercase flex items-center gap-2"><i data-lucide="pie-chart" class="w-5 h-5 text-[var(--color-primary)]"></i> BALANCE DEL MES</h4><div class="w-48 h-48 mb-4"><canvas id="financeDoughnutChart"></canvas></div></div>
                        <div class="lg:col-span-2"><h4 class="text-lg font-extrabold text-gray-700 mb-4 flex items-center gap-2"><i data-lucide="list-checks" class="w-5 h-5 text-[var(--color-primary)]"></i> TRANSACCIONES DEL MES</h4><div id="transaction-list" class="space-y-3 max-h-[400px] overflow-y-auto pr-2"></div></div>
                    </div>

                    <div class="mt-8">
                        <h4 class="text-xl font-extrabold text-gray-800 mb-4 flex items-center gap-2"><i data-lucide="table" class="w-6 h-6 text-[var(--color-primary)]"></i> RESUMEN HISTÓRICO</h4>
                        <div class="overflow-x-auto">
                            <table class="min-w-full bg-white rounded-lg shadow-sm overflow-hidden responsive-table-stack">
                                <thead class="bg-gray-50"><tr class="text-left text-gray-500 uppercase text-sm"><th class="py-3 px-6 font-semibold">MES</th><th class="py-3 px-6 font-semibold text-right">INGRESOS</th><th class="py-3 px-6 font-semibold text-right">GASTOS</th><th class="py-3 px-6 font-semibold text-right">GANANCIA NETA</th></tr></thead>
                                <tbody class="text-gray-700 text-sm" id="finance-summary-table-body"></tbody>
                            </table>
                        </div>
                    </div>

                    <div class="mt-8">
                        <h4 class="text-xl font-extrabold text-gray-800 mb-4 flex items-center gap-2">
                            <i data-lucide="repeat" class="w-6 h-6 text-[var(--color-primary)]"></i> GESTIÓN DE GASTOS RECURRENTES
                        </h4>
                        <div class="overflow-x-auto">
                            <table class="min-w-full bg-white rounded-lg shadow-sm overflow-hidden responsive-table-stack">
                                <thead class="bg-gray-50">
                                    <tr class="text-left text-gray-500 uppercase text-sm">
                                        <th class="py-3 px-6 font-semibold">DESCRIPCIÓN</th>
                                        <th class="py-3 px-6 font-semibold text-right">MONTO</th>
                                        <th class="py-3 px-6 font-semibold text-center">DÍA DE PAGO</th>
                                        <th class="py-3 px-6 font-semibold">CLIENTE</th>
                                        <th class="py-3 px-6 font-semibold text-center">ACCIONES</th>
                                    </tr>
                                </thead>
                                <tbody class="text-gray-700 text-sm" id="recurring-expenses-table-body">
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </section>
        </div>
    </main>
</div>

<section id="addFinancesPanel" class="hidden">
    <div id="finances-panel-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-40"></div>
    <div id="finances-panel" class="fixed inset-y-0 right-0 w-full max-w-md bg-gray-100 shadow-xl z-50 transform translate-x-full transition-transform duration-300 ease-in-out">
        <div class="bg-gray-100 p-6 overflow-y-auto space-y-6 h-full">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-2xl font-bold text-[var(--color-primary)]">REGISTRAR TRANSACCIÓN</h3>
                <button onclick="hideFinancesPanel()" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
            </div>
            <form id="add-transaccion-form">
                <div class="mb-4 relative"><label for="transaccion-cliente-display" class="block text-gray-700 text-sm font-bold mb-2 uppercase">CLIENTE (OPCIONAL)</label><input type="text" id="transaccion-cliente-display" placeholder="Buscar Cliente..." class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white"><input type="hidden" id="transaccion-cliente"><div id="transaccion-cliente-suggestions" class="autocomplete-suggestions hidden"></div><p id="transaccion-cliente-info" class="text-xs text-gray-500 mt-1"></p></div>
                <div class="mb-4"><label for="transaccion-tipo" class="block text-gray-700 text-sm font-bold mb-2 uppercase">TIPO</label><select id="transaccion-tipo" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white"><option value="ingreso">INGRESO</option><option value="gasto">GASTO</option></select></div>
                <div id="recurring-options-container" class="mb-4 hidden"><div class="flex items-center mb-2"><input type="checkbox" id="is-recurring-expense" class="h-4 w-4 text-[var(--color-primary)] border-gray-300 rounded focus:ring-[var(--color-highlight)]"><label for="is-recurring-expense" class="ml-2 block text-gray-700 text-sm font-bold uppercase">GASTO RECURRENTE (TODOS LOS MESES)</label></div><div id="recurrence-day-group" class="mb-4 hidden"><label for="recurrence-day" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DÍA DEL MES (1-31)</label><input type="number" id="recurrence-day" min="1" max="31" value="1" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]"></div></div>
                <div class="mb-4"><label for="transaccion-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label><input type="text" id="transaccion-descripcion" class="w-full p-3 border border-gray-300 rounded-lg" required></div>
                <div class="mb-4"><label for="transaccion-monto" class="block text-gray-700 text-sm font-bold mb-2 uppercase">MONTO</label><input type="number" step="0.01" id="transaccion-monto" class="w-full p-3 border border-gray-300 rounded-lg" required></div>
                <div id="recurrence-start-date-group-add" class="mb-4"><label for="transaccion-fecha" class="block text-gray-700 text-sm font-bold mb-2 uppercase" id="transaccion-fecha-label-add">FECHA</label><input type="date" id="transaccion-fecha" class="w-full p-3 border border-gray-300 rounded-lg" required></div>
                <div id="recurrence-end-date-group-add" class="mb-6 hidden"><label for="transaccion-end-date" class="block text-gray-700 text-sm font-bold mb-2 uppercase">FECHA DE FIN (OPCIONAL)</label><input type="date" id="transaccion-end-date" class="w-full p-3 border border-gray-300 rounded-lg"></div>
                <div class="flex flex-col sm:flex-row justify-end space-y-2 sm:space-y-0 sm:space-x-4 mt-4">
                    <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase w-full sm:w-auto" onclick="hideFinancesPanel()">CANCELAR</button>
                    <button type="submit" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase w-full sm:w-auto">GUARDAR</button>
                </div>
            </form>
        </div>
    </div>
</section>

<section id="editFinancesPanel" class="hidden">
    <div id="edit-finances-panel-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-40"></div>
    <div id="edit-finances-panel" class="fixed inset-y-0 right-0 w-full max-w-md bg-gray-100 shadow-xl z-50 transform translate-x-full transition-transform duration-300 ease-in-out">
        <div class="bg-gray-100 p-6 overflow-y-auto space-y-6 h-full">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-2xl font-bold text-[var(--color-primary)]">EDITAR TRANSACCIÓN</h3>
                <button onclick="hideEditFinancesPanel()" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
            </div>
            <form id="view-transaccion-form">
                <input type="hidden" id="view-transaccion-id">
                <input type="hidden" id="currentTransactionIsRecurring">
                <div class="mb-4 relative"><label for="view-transaccion-cliente-display" class="block text-gray-700 text-sm font-bold mb-2 uppercase">CLIENTE</label><input type="text" id="view-transaccion-cliente-display" placeholder="Buscar Cliente..." class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white"><input type="hidden" id="view-transaccion-cliente"><div id="view-transaccion-cliente-suggestions" class="autocomplete-suggestions hidden"></div><p id="view-transaccion-cliente-info" class="text-xs text-gray-500 mt-1"></p></div>
                <div class="mb-4"><label for="view-transaccion-tipo" class="block text-gray-700 text-sm font-bold mb-2 uppercase">TIPO</label><select id="view-transaccion-tipo" class="w-full p-3 border border-gray-300 rounded-lg bg-white"><option value="ingreso">INGRESO</option><option value="gasto">GASTO</option></select></div>
                <div id="view-recurring-options-container" class="mb-4 hidden"><div class="flex items-center mb-2"><input type="checkbox" id="view-is-recurring-expense" class="h-4 w-4 text-[var(--color-primary)] border-gray-300 rounded focus:ring-[var(--color-highlight)]" disabled><label for="view-is-recurring-expense" class="ml-2 block text-gray-700 text-sm font-bold uppercase">GASTO RECURRENTE</label></div><div id="view-recurrence-day-group" class="mb-4 hidden"><label for="view-recurrence-day" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DÍA DEL MES (1-31)</label><input type="number" id="view-recurrence-day" min="1" max="31" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]"></div><div id="view-recurrence-start-date-group" class="mb-4 hidden"><label for="view-recurrence-start-date" class="block text-gray-700 text-sm font-bold mb-2 uppercase">FECHA DE INICIO</label><input type="date" id="view-recurrence-start-date" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]"></div><div id="view-recurrence-end-date-group" class="mb-4 hidden"><label for="view-recurrence-end-date" class="block text-gray-700 text-sm font-bold mb-2 uppercase">FECHA DE FIN (OPCIONAL)</label><input type="date" id="view-transaccion-end-date" class="w-full p-3 border border-gray-300 rounded-lg"></div></div>
                <div class="mb-4"><label for="view-transaccion-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label><input type="text" id="view-transaccion-descripcion" class="w-full p-3 border border-gray-300 rounded-lg" required></div>
                <div class="mb-4"><label for="view-transaccion-monto" class="block text-gray-700 text-sm font-bold mb-2 uppercase">MONTO</label><input type="number" step="0.01" id="view-transaccion-monto" class="w-full p-3 border border-gray-300 rounded-lg" required></div>
                <div class="mb-6"><label for="view-transaccion-fecha" class="block text-gray-700 text-sm font-bold mb-2 uppercase">FECHA</label><input type="date" id="view-transaccion-fecha" class="w-full p-3 border border-gray-300 rounded-lg" required></div>
                <div class="flex flex-col sm:flex-row justify-end space-y-2 sm:space-y-0 sm:space-x-4 mt-4">
                    <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase w-full sm:w-auto" onclick="hideEditFinancesPanel()">CANCELAR</button>
                    <button type="submit" class="btn-primary font-bold py-2 px-4 rounded-lg uppercase w-full sm:w-auto" id="save-transaction-changes-btn">GUARDAR CAMBIOS</button>
                </div>
            </form>
        </div>
    </div>
</section>

<div id="confirmDeleteModal" class="fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center hidden z-50">
    <div class="bg-white p-8 rounded-xl shadow-2xl w-full max-w-sm m-4 text-center">
        <div class="flex justify-center mb-4"><i data-lucide="alert-triangle" class="w-16 h-16 text-red-500"></i></div>
        <h3 class="text-2xl font-bold text-[var(--color-primary)] mb-4 uppercase">CONFIRMAR ELIMINACIÓN</h3>
        <p class="text-gray-700 mb-6 uppercase">¿ESTÁS SEGURO DE QUE DESEAS ELIMINAR ESTE <span id="confirm-item-type" class="font-semibold">ELEMENTO</span>? ESTA ACCIÓN NO SE PUEDE DESHACER.</p>
        <div class="flex flex-col sm:flex-row justify-center space-y-2 sm:space-y-0 sm:space-x-4">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase w-full sm:w-auto" onclick="closeModal('confirmDeleteModal')">CANCELAR</button>
            <button type="button" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase w-full sm:w-auto" id="confirm-delete-button">CONFIRMAR</button>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    // --- VARIABLES GLOBALES ---
    let allTransactions = <?php echo $transactions_json; ?>;
    let allRecurringExpenses = <?php echo $recurring_expenses_json; ?>;
    let allClients = [];

    // --- VARIABLES DOM ---
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const sidebar = document.getElementById('sidebar');
    const sidebarOverlay = document.getElementById('sidebar-overlay');
    const confirmDeleteButton = document.getElementById('confirm-delete-button');
    const confirmItemTypeSpan = document.getElementById('confirm-item-type');
    const financeMonthSelector = document.getElementById('finance-month-selector');
    const transactionList = document.getElementById('transaction-list');
    const financeDoughnutChart = document.getElementById('financeDoughnutChart');
    const addTransaccionForm = document.getElementById('add-transaccion-form');
    const viewTransaccionForm = document.getElementById('view-transaccion-form');
    const financeSummaryTableBody = document.getElementById('finance-summary-table-body');
    const recurringExpensesTableBody = document.getElementById('recurring-expenses-table-body');
    const transaccionTipo = document.getElementById('transaccion-tipo');
    const recurringOptionsContainer = document.getElementById('recurring-options-container');
    const isRecurringExpenseCheckbox = document.getElementById('is-recurring-expense');
    const transaccionFechaInput = document.getElementById('transaccion-fecha');
    const transaccionFechaLabelAdd = document.getElementById('transaccion-fecha-label-add');
    const recurrenceEndDateGroupAdd = document.getElementById('recurrence-end-date-group-add');
    const recurrenceDayGroup = document.getElementById('recurrence-day-group');
    const recurrenceDayInput = document.getElementById('recurrence-day');
    const financesPanelOverlay = document.getElementById('finances-panel-overlay');
    const financesPanel = document.getElementById('finances-panel');
    const editFinancesPanel = document.getElementById('editFinancesPanel');
    const editFinancesPanelInner = document.getElementById('edit-finances-panel');
    const editFinancesPanelOverlay = document.getElementById('edit-finances-panel-overlay');

    let financeChartInstance;

    // --- FUNCIONES GLOBALES Y DE UI ---
    window.showFinancesPanel = () => {
        addFinancesPanel.classList.remove('hidden');
        setTimeout(() => { financesPanel.classList.remove('translate-x-full'); }, 10);
        
        addTransaccionForm.reset();
        document.getElementById('transaccion-cliente-display').value = ''; // Clear autocomplete display
        document.getElementById('transaccion-cliente').value = ''; // Clear hidden ID
        document.getElementById('transaccion-cliente-info').textContent = ''; // Clear info text

        transaccionTipo.value = 'ingreso';
        recurringOptionsContainer.classList.add('hidden');
        isRecurringExpenseCheckbox.checked = false; // Reset checkbox
        recurrenceDayGroup.classList.add('hidden');
        recurrenceEndDateGroupAdd.classList.add('hidden');
        transaccionFechaLabelAdd.textContent = 'FECHA';
        transaccionFechaInput.value = new Date().toISOString().slice(0, 10);

        financesPanelOverlay.classList.remove('hidden');
        document.body.style.overflow = 'hidden';
    };

    window.hideFinancesPanel = () => {
        financesPanel.classList.add('translate-x-full');
        setTimeout(() => {
            addFinancesPanel.classList.add('hidden');
            financesPanelOverlay.classList.add('hidden');
            document.body.style.overflow = '';
            // Reload data after panel closes. Only show toast if successfully saved.
            // This toast is now handled by the form submit success/error.
            // reloadData(); 
        }, 300);
    };

    window.showEditFinancesPanel = () => {
        editFinancesPanel.classList.remove('hidden');
        setTimeout(() => { editFinancesPanelInner.classList.remove('translate-x-full'); }, 10);
        editFinancesPanelOverlay.classList.remove('hidden');
        document.body.style.overflow = 'hidden';
    };

    window.hideEditFinancesPanel = () => {
        editFinancesPanelInner.classList.add('translate-x-full');
        setTimeout(() => {
            editFinancesPanel.classList.add('hidden');
            editFinancesPanelOverlay.classList.add('hidden');
            document.body.style.overflow = '';
            // Reload data after panel closes. Only show toast if successfully saved.
            // This toast is now handled by the form submit success/error.
            // reloadData(); 
        }, 300);
    };
    
    window.openModal = (modalId) => {
        const modal = document.getElementById(modalId);
        if (!modal) { showToast(`Error: Modal con ID "${modalId}" no encontrado.`, 'error'); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        modal.classList.remove('hidden');
        setTimeout(() => { 
            modalBox.classList.remove('scale-95', 'opacity-0'); 
            // Add scale and opacity for smooth transition when opening
            modalBox.style.transform = 'scale(1)';
            modalBox.style.opacity = '1';
        }, 50);
        document.body.style.overflow = 'hidden';
    };

    window.closeModal = (modalId) => {
        const modal = document.getElementById(modalId);
        if (!modal) { showToast(`Error: Modal con ID "${modalId}" no encontrado para cerrar.`, 'error'); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        modalBox.classList.add('scale-95', 'opacity-0');
        modalBox.style.transform = 'scale(0.95)'; // Reset transform for next open
        modalBox.style.opacity = '0'; // Reset opacity for next open

        setTimeout(() => {  
            modal.classList.add('hidden');  
            document.body.style.overflow = '';
        }, 300);
    };

    window.openConfirmDeleteModal = (itemId, itemType) => {
        confirmDeleteButton.dataset.itemId = itemId;
        if (itemType === 'recurrente') {
            confirmItemTypeSpan.textContent = "GASTO RECURRENTE";
            confirmDeleteButton.onclick = () => deleteRecurringExpense(itemId);
        } else {
            confirmItemTypeSpan.textContent = "TRANSACCIÓN";
            confirmDeleteButton.onclick = () => deleteTransaction(itemId);
        }
        window.openModal('confirmDeleteModal');
    };

    window.openViewTransaccionModal = (transactionId, isRecurringTransaction = false) => {
        let transaction;
        if (isRecurringTransaction) {
            transaction = allRecurringExpenses.find(t => t.id == transactionId);
        } else {
            transaction = allTransactions.find(t => t.id == transactionId);
        }
        if (!transaction) {
            showToast('Transacción no encontrada.', 'error');
            return;
        }
        
        document.getElementById('view-transaccion-id').value = transaction.id;
        document.getElementById('view-transaccion-tipo').value = transaction.type;
        document.getElementById('view-transaccion-descripcion').value = transaction.description;
        document.getElementById('view-transaccion-monto').value = transaction.amount;
        document.getElementById('view-transaccion-fecha').value = transaction.transaction_date || transaction.start_date;
        document.getElementById('currentTransactionIsRecurring').value = isRecurringTransaction;
        
        // Handle client autocomplete display for view/edit panel
        const viewTransaccionClienteDisplay = document.getElementById('view-transaccion-cliente-display');
        const viewTransaccionClienteInput = document.getElementById('view-transaccion-cliente');
        const viewTransaccionClienteInfo = document.getElementById('view-transaccion-cliente-info');

        if (transaction.client_id) {
            const client = allClients.find(c => c.id == transaction.client_id);
            viewTransaccionClienteDisplay.value = client ? client.name : '';
            viewTransaccionClienteInput.value = transaction.client_id;
            viewTransaccionClienteInfo.textContent = client ? `Email: ${client.email || 'N/A'} | Tel: ${formatPhoneNumber(client.phone) || formatPhoneNumber(client.mobile) || 'N/A'}` : '';
        } else {
            viewTransaccionClienteDisplay.value = '';
            viewTransaccionClienteInput.value = '';
            viewTransaccionClienteInfo.textContent = '';
        }

        // --- Recurrence fields visibility for edit panel ---
        const viewRecurringOptionsContainer = document.getElementById('view-recurring-options-container');
        const viewRecurrenceDayGroup = document.getElementById('view-recurrence-day-group');
        const viewRecurrenceStartDateGroup = document.getElementById('view-recurrence-start-date-group');
        const viewRecurrenceEndDateGroup = document.getElementById('view-recurrence-end-date-group');
        const viewTransaccionFecha = document.getElementById('view-transaccion-fecha'); // The single transaction date field

        if (isRecurringTransaction) {
            viewRecurringOptionsContainer.classList.remove('hidden');
            viewRecurrenceDayGroup.classList.remove('hidden');
            viewRecurrenceStartDateGroup.classList.remove('hidden');
            viewRecurrenceEndDateGroup.classList.remove('hidden');
            viewTransaccionFecha.parentElement.classList.add('hidden'); // Hide the single date field
            
            document.getElementById('view-recurrence-day').value = transaction.recurrence_day_of_month;
            document.getElementById('view-recurrence-start-date').value = transaction.start_date;
            document.getElementById('view-transaccion-end-date').value = transaction.end_date || '';
        } else {
            viewRecurringOptionsContainer.classList.add('hidden');
            viewRecurrenceDayGroup.classList.add('hidden');
            viewRecurrenceStartDateGroup.classList.add('hidden');
            viewRecurrenceEndDateGroup.classList.add('hidden');
            viewTransaccionFecha.parentElement.classList.remove('hidden'); // Show the single date field
        }


        window.showEditFinancesPanel();
    };

    async function reloadData() {
        location.reload();
    }

    // --- FUNCIONES DE RENDERIZADO ---
    
    function setupFinanceMonthSelector() {
        const months = new Set();
        allTransactions.forEach(t => months.add(t.transaction_date.substring(0, 7)));
        if(months.size === 0) months.add(new Date().toISOString().substring(0, 7));
        const sortedMonths = Array.from(months).sort().reverse();
        financeMonthSelector.innerHTML = '';
        sortedMonths.forEach(month => {
            const option = document.createElement('option');
            option.value = month;
            const [year, monthNum] = month.split('-');
            option.textContent = new Date(year, monthNum - 1, 1).toLocaleString('es-ES', { month: 'long', year: 'numeric' }).toUpperCase();
            financeMonthSelector.appendChild(option);
        });
        if (sortedMonths.length > 0) financeMonthSelector.value = sortedMonths[0];
    }

    function renderFinances() {
        const selectedMonth = financeMonthSelector.value;
        const monthlyTransactions = allTransactions.filter(t => t.transaction_date.startsWith(selectedMonth));
        let totalIncome = 0, totalExpense = 0;
        
        monthlyTransactions.forEach(t => {
            if (t.type === 'ingreso') totalIncome += parseFloat(t.amount);
            else totalExpense += parseFloat(t.amount);
        });
        
        transactionList.innerHTML = '';
        if (monthlyTransactions.length === 0) {
            transactionList.innerHTML = `<p class="text-gray-500 text-center uppercase">NO HAY TRANSACCIONES PARA ESTE MES.</p>`;
        } else {
            monthlyTransactions.forEach(t => {
                const isIncome = t.type === 'ingreso';
                const clientInfo = t.client_name ? `<p class="text-xs text-blue-600 font-semibold">${t.client_name}</p>` : '';
                const card = document.createElement('div');
                card.className = `transaction-card flex items-center p-3 rounded-lg group ${isIncome ? 'bg-green-50' : 'bg-red-50'}`;
                card.innerHTML = `
                    <div class="rounded-full p-2 mr-4 ${isIncome ? 'bg-green-200' : 'bg-red-200'}"><i data-lucide="${isIncome ? 'arrow-up-right' : 'arrow-down-left'}" class="w-5 h-5 ${isIncome ? 'text-green-700' : 'text-red-700'}"></i></div>
                    <div class="flex-grow"><p class="font-bold text-gray-800 uppercase">${t.description}</p>${clientInfo}<p class="text-sm text-gray-500">${t.transaction_date}</p></div>
                    <div class="text-right">
                        <p class="font-bold ${isIncome ? 'text-green-600' : 'text-red-600'}">${isIncome ? '+' : '-'}$${parseFloat(t.amount).toFixed(2)}</p>
                        <div class="flex items-center justify-center space-x-2 mt-1 opacity-0 group-hover:opacity-100 transition-opacity">
                            <button class="text-gray-400 hover:text-blue-600" onclick="window.openViewTransaccionModal(${t.id}, false)"><i data-lucide="edit-2" class="w-4 h-4"></i></button>
                            <button class="text-gray-400 hover:text-red-600" onclick="window.openConfirmDeleteModal(${t.id}, 'transaccion')"><i data-lucide="trash-2" class="w-4 h-4"></i></button>
                        </div>
                    </div>`;
                transactionList.appendChild(card);
            });
        }
        
        document.getElementById('finance-total-income').textContent = `$${totalIncome.toFixed(2)}`;
        document.getElementById('finance-total-expense').textContent = `$${totalExpense.toFixed(2)}`;
        document.getElementById('finance-net-profit').textContent = `$${(totalIncome - totalExpense).toFixed(2)}`;
        
        if (financeChartInstance) financeChartInstance.destroy();
        financeChartInstance = new Chart(document.getElementById('financeDoughnutChart').getContext('2d'), {
            type: 'doughnut',
            data: { datasets: [{ data: [totalIncome || 0.01, totalExpense || 0.01], backgroundColor: ['#16a34a', '#dc2626'], borderColor: '#fff', borderWidth: 4 }] },
            options: { responsive: true, maintainAspectRatio: false, cutout: '70%', plugins: { legend: { display: false } } }
        });
        renderFinanceSummaryTable();
        lucide.createIcons();
    }
    
    function renderFinanceSummaryTable() {
        const monthlySummaries = {};
        allTransactions.forEach(t => {
            const month = t.transaction_date.substring(0, 7);
            if (!monthlySummaries[month]) monthlySummaries[month] = { income: 0, expense: 0 };
            if (t.type === 'ingreso') monthlySummaries[month].income += parseFloat(t.amount);
            else monthlySummaries[month].expense += parseFloat(t.amount);
        });
        const sortedMonths = Object.keys(monthlySummaries).sort().reverse();
        financeSummaryTableBody.innerHTML = '';
        sortedMonths.forEach(month => {
            const summary = monthlySummaries[month];
            const net = summary.income - summary.expense;
            const monthName = new Date(month + '-02T00:00:00').toLocaleString('es-ES', { month: 'long', year: 'numeric' }).toUpperCase();
            const row = financeSummaryTableBody.insertRow();
            // Added data-label attributes for responsive stacking
            row.innerHTML = `
                <td class="py-4 px-6 font-semibold" data-label="MES">${monthName}</td>
                <td class="py-4 px-6 text-right text-green-600 font-bold" data-label="INGRESOS">$${summary.income.toFixed(2)}</td>
                <td class="py-4 px-6 text-right text-red-600 font-bold" data-label="GASTOS">$${summary.expense.toFixed(2)}</td>
                <td class="py-4 px-6 text-right font-bold ${net >= 0 ? 'text-blue-700' : 'text-red-700'}" data-label="GANANCIA NETA">$${net.toFixed(2)}</td>
            `;
            financeSummaryTableBody.appendChild(row);
        });
    }

    function renderRecurringExpensesTable() {
        recurringExpensesTableBody.innerHTML = '';
        if (allRecurringExpenses.length === 0) {
            recurringExpensesTableBody.innerHTML = `<tr><td colspan="5" class="text-center py-4 text-gray-500 uppercase">No tienes gastos recurrentes configurados.</td></tr>`;
            return;
        }
        allRecurringExpenses.forEach(exp => {
            const row = recurringExpensesTableBody.insertRow();
            row.className = 'border-b border-gray-200 hover:bg-gray-50';
            const clientName = exp.client_name || '<span class="text-gray-400">N/A</span>';
            // Added data-label attributes for responsive stacking
            row.innerHTML = `
                <td class="py-4 px-6 font-semibold" data-label="DESCRIPCIÓN">${exp.description}</td>
                <td class="py-4 px-6 text-right font-bold text-red-600" data-label="MONTO">$${parseFloat(exp.amount).toFixed(2)}</td>
                <td class="py-4 px-6 text-center" data-label="DÍA DE PAGO">${exp.recurrence_day_of_month}</td>
                <td class="py-4 px-6" data-label="CLIENTE">${clientName}</td>
                <td class="py-4 px-6 text-center actions-cell" data-label="ACCIONES">
                    <div class="flex flex-col sm:flex-row items-center justify-center space-y-2 sm:space-y-0 sm:space-x-2">
                        <button class="text-gray-400 hover:text-blue-600 w-full sm:w-auto" title="Editar Regla" onclick="window.openViewTransaccionModal(${exp.id}, true)"><i data-lucide="edit-2" class="w-4 h-4"></i></button>
                        <button class="text-gray-400 hover:text-red-600 w-full sm:w-auto" title="Eliminar Regla Permanentemente" onclick="window.openConfirmDeleteModal(${exp.id}, 'recurrente')"><i data-lucide="trash-2" class="w-4 h-4"></i></button>
                    </div>
                </td>`;
            recurringExpensesTableBody.appendChild(row);
        });
        lucide.createIcons();
    }
    
    // --- LÓGICA DE INTERACCIÓN CON EL SERVIDOR ---
    async function deleteTransaction(transactionId) {
        try {
            const response = await fetch('db/finances-delete.php', {
                method: 'DELETE',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({ id: transactionId })
            });
            const result = await response.json();
            if (result.success) {
                closeModal('confirmDeleteModal');
                showToast('Transacción eliminada con éxito.', 'success'); // Added toast
                await reloadData();
            } else {
                showToast(result.message || 'Error al eliminar transacción.', 'error'); // Added toast
            }
        } catch (error) {
            showToast('Error de conexión al eliminar transacción.', 'error'); // Added toast
        }
    }

    async function deleteRecurringExpense(recurringExpenseId) {
        try {
            const response = await fetch('db/recurring-finances-delete.php', {
                method: 'DELETE',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({ id: recurringExpenseId })
            });
            const result = await response.json();
            if (result.success) {
                closeModal('confirmDeleteModal');
                showToast('Gasto recurrente eliminado con éxito.', 'success'); // Added toast
                await reloadData();
            } else {
                showToast(result.message || 'Error al eliminar gasto recurrente.', 'error'); // Added toast
            }
        } catch (error) {
            showToast('Error de conexión al eliminar gasto recurrente.', 'error'); // Added toast
        }
    }

    async function fetchAllClients() {
        try {
            const response = await fetch('db/clients-read-contact-data.php');
            const result = await response.json();
            if (result.success) {
                allClients = result.data;
            } else {
                showToast("Error al cargar clientes: " + (result.message || 'Error desconocido.'), 'error'); // Added toast
            }
        } catch (error) {
            showToast("Error de conexión al cargar clientes.", 'error'); // Added toast
        }
    }

    // --- INICIALIZACIÓN Y EVENT LISTENERS ---
    setupFinanceMonthSelector();
    renderFinances();
    renderRecurringExpensesTable();
    
    if (financeMonthSelector) {
        financeMonthSelector.addEventListener('change', renderFinances);
    }
    
    // Lógica para mostrar/ocultar campos de recurrencia en el formulario
    transaccionTipo.addEventListener('change', function() {
        if (this.value === 'gasto') {
            recurringOptionsContainer.classList.remove('hidden');
        } else {
            recurringOptionsContainer.classList.add('hidden');
            isRecurringExpenseCheckbox.checked = false;
            isRecurringExpenseCheckbox.dispatchEvent(new Event('change')); // Trigger change to hide sub-options
        }
    });

    isRecurringExpenseCheckbox.addEventListener('change', function() {
        if (this.checked) {
            recurrenceDayGroup.classList.remove('hidden');
            recurrenceEndDateGroupAdd.classList.remove('hidden');
            transaccionFechaLabelAdd.textContent = 'FECHA DE INICIO';
            syncRecurrenceDay(); // Sincroniza el día al marcar
        } else {
            recurrenceDayGroup.classList.add('hidden');
            recurrenceEndDateGroupAdd.classList.add('hidden');
            transaccionFechaLabelAdd.textContent = 'FECHA';
        }
    });
    
    // Lógica para sincronizar el día del mes con la fecha de inicio
    function syncRecurrenceDay() {
        const dateValue = transaccionFechaInput.value;
        if (dateValue) {
            const day = parseInt(dateValue.split('-')[2], 10);
            recurrenceDayInput.value = day;
        }
    }
    transaccionFechaInput.addEventListener('change', function() {
        if (isRecurringExpenseCheckbox.checked) {
            syncRecurrenceDay();
        }
    });

    // Listener para el formulario de añadir transacción
    addTransaccionForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const type = document.getElementById('transaccion-tipo').value;
        const isRecurring = document.getElementById('is-recurring-expense').checked;
        const recurrenceDayValue = document.getElementById('recurrence-day').value;

        const transactionData = {
            client_id: document.getElementById('transaccion-cliente').value || null,
            type: type,
            description: document.getElementById('transaccion-descripcion').value,
            amount: document.getElementById('transaccion-monto').value,
            transaction_date: document.getElementById('transaccion-fecha').value,
            is_recurring: isRecurring && type === 'gasto',
            recurrence_day_of_month: (isRecurring && type === 'gasto') ? recurrenceDayValue : null,
            end_date: (isRecurring && type === 'gasto') ? document.getElementById('transaccion-end-date').value || null : null
        };
        
        try {
            const response = await fetch('db/finances-create.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(transactionData)
            });
            const result = await response.json();
            if (result.success) {
                hideFinancesPanel();
                showToast('Transacción agregada con éxito.', 'success'); // Added toast
                await reloadData();
            } else {
                showToast(result.message || 'Error al guardar.', 'error'); // Added toast
            }
        } catch (error) {
            showToast('Error de conexión al guardar.', 'error'); // Added toast
        }
    });
    
    // Listener para el formulario de editar transacción
    viewTransaccionForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const transactionId = document.getElementById('view-transaccion-id').value;
        const currentIsRecurring = document.getElementById('currentTransactionIsRecurring').value === 'true'; // Get original recurring status
        const type = document.getElementById('view-transaccion-tipo').value;
        
        let updateUrl = 'db/finances-update.php';
        let updateData = {
            id: transactionId,
            description: document.getElementById('view-transaccion-descripcion').value,
            amount: document.getElementById('view-transaccion-monto').value,
            transaction_date: document.getElementById('view-transaccion-fecha').value,
            type: type,
            client_id: document.getElementById('view-transaccion-cliente').value || null
        };

        if (currentIsRecurring) { // If it was originally recurring, it's a recurring_finances update
            updateUrl = 'db/recurring-finances-update.php';
            updateData.recurrence_day_of_month = document.getElementById('view-recurrence-day').value;
            updateData.start_date = document.getElementById('view-recurrence-start-date').value;
            updateData.end_date = document.getElementById('view-transaccion-end-date').value || null;
        }

        try {
            const response = await fetch(updateUrl, {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(updateData)
            });
            const result = await response.json();
            if (result.success) {
                hideEditFinancesPanel();
                showToast('Cambios guardados con éxito.', 'success'); // Added toast
                await reloadData();
            } else {
                showToast(result.message || 'Error al guardar cambios.', 'error'); // Added toast
            }
        } catch (error) {
            showToast('Error de conexión al guardar cambios.', 'error'); // Added toast
        }
    });


    // Autocomplete para clientes en los formularios de finanzas
    const setupClientAutocomplete = (displayInput, hiddenInput, suggestionsDiv, infoDisplay) => {
        let currentClientData = {}; // Para almacenar el cliente seleccionado
        displayInput.addEventListener('input', async function() {
            const searchTerm = this.value.toLowerCase();
            suggestionsDiv.innerHTML = '';
            hiddenInput.value = ''; // Limpiar ID oculto si el usuario escribe de nuevo
            infoDisplay.textContent = ''; // Limpiar info si el usuario escribe de nuevo

            if (searchTerm.length < 2) {
                suggestionsDiv.classList.add('hidden');
                return;
            }

            const filteredClients = Object.values(allClients).filter(client =>
                client.name.toLowerCase().includes(searchTerm) ||
                (client.email && client.email.toLowerCase().includes(searchTerm))
            );

            if (filteredClients.length > 0) {
                filteredClients.forEach(client => {
                    const suggestionItem = document.createElement('div');
                    suggestionItem.className = 'autocomplete-suggestion-item p-2 cursor-pointer hover:bg-gray-100 border-b border-gray-200 last:border-b-0';
                    suggestionItem.textContent = `${client.name} (${client.email || 'N/A'})`;
                    suggestionItem.addEventListener('click', () => {
                        displayInput.value = client.name;
                        hiddenInput.value = client.id;
                        infoDisplay.textContent = `Email: ${client.email || 'N/A'} | Tel: ${formatPhoneNumber(client.phone) || formatPhoneNumber(client.mobile) || 'N/A'}`;
                        suggestionsDiv.classList.add('hidden');
                        currentClientData = client; // Guarda el cliente seleccionado
                        showToast(`Cliente "${client.name}" seleccionado.`, 'info'); // Added toast
                    });
                    suggestionsDiv.appendChild(suggestionItem);
                });
                suggestionsDiv.classList.remove('hidden');
            } else {
                suggestionsDiv.classList.add('hidden');
            }
        });

        displayInput.addEventListener('focus', function() {
            if (this.value.length >= 2) {
                this.dispatchEvent(new Event('input')); // Muestra sugerencias al enfocar si ya hay texto
            }
        });

        displayInput.addEventListener('blur', () => {
            // Espera un poco para que el click en una sugerencia se registre
            setTimeout(() => { suggestionsDiv.classList.add('hidden'); }, 150);
        });
    };


    // Inicializar autocompletado para el formulario de añadir transacción
    const addTransaccionClienteDisplay = document.getElementById('transaccion-cliente-display');
    const addTransaccionClienteInput = document.getElementById('transaccion-cliente');
    const addTransaccionClienteSuggestions = document.getElementById('transaccion-cliente-suggestions');
    const addTransaccionClienteInfo = document.getElementById('transaccion-cliente-info');
    setupClientAutocomplete(addTransaccionClienteDisplay, addTransaccionClienteInput, addTransaccionClienteSuggestions, addTransaccionClienteInfo);

    // Inicializar autocompletado para el formulario de editar transacción
    const viewTransaccionClienteDisplay = document.getElementById('view-transaccion-cliente-display');
    const viewTransaccionClienteInput = document.getElementById('view-transaccion-cliente');
    const viewTransaccionClienteSuggestions = document.getElementById('view-transaccion-cliente-suggestions');
    const viewTransaccionClienteInfo = document.getElementById('view-transaccion-cliente-info');
    setupClientAutocomplete(viewTransaccionClienteDisplay, viewTransaccionClienteInput, viewTransaccionClienteSuggestions, viewTransaccionClienteInfo);

    // Sidebar overlay
    if (mobileMenuButton) mobileMenuButton.addEventListener('click', () => {
        sidebar.classList.toggle('-translate-x-full');
        sidebarOverlay.classList.toggle('hidden');
    });
    if (sidebarOverlay) sidebarOverlay.addEventListener('click', () => {
        sidebar.classList.toggle('-translate-x-full');
        sidebarOverlay.classList.toggle('hidden');
        // Close finance panels if sidebar overlay is clicked
        if (!financesPanel.classList.contains('translate-x-full')) {
            hideFinancesPanel();
        }
        if (!editFinancesPanelInner.classList.contains('translate-x-full')) {
            hideEditFinancesPanel();
        }
    });

    // Ocultar overlays al hacer clic en ellos (para los paneles de finanzas/edición)
    financesPanelOverlay.addEventListener('click', hideFinancesPanel);
    editFinancesPanelOverlay.addEventListener('click', hideEditFinancesPanel);


    fetchAllClients(); // Cargar clientes al inicio para el autocompletado

    // --- Ejecución inicial de renderizado ---
    setupFinanceMonthSelector();
    renderFinances();
    renderRecurringExpensesTable(); // Render this table as well

    lucide.createIcons();


    function formatPhoneNumber(phoneNumberString) {
        if (!phoneNumberString) return '';
        const cleaned = ('' + phoneNumberString).replace(/\D/g, '');
        const match = cleaned.match(/^(\d{3})(\d{3})(\d{4})$/);
        // This is a US format. Based on previous context, user is in Guatemala.
        // It's important to clarify if numbers should be formatted for US or Guatemala.
        // For now, keeping the existing formatting which is for US numbers.
        return match ? `(${match[1]}) ${match[2]}-${match[3]}` : phoneNumberString;
    }
    
    function descargarComoCSV(data, filename) {
        const headers = Object.keys(data[0]);
        const csvRows = [headers.map(h => `"${h}"`).join(',')]; // Add quotes to headers
        for (const row of data) {
            const values = headers.map(header => {
                const value = String(row[header] || '');
                // Escape double quotes by doubling them, then wrap in double quotes
                return `"${value.replace(/"/g, '""')}"`;
            });
            csvRows.push(values.join(','));
        }
        // Add BOM for proper UTF-8 encoding in Excel
        const blob = new Blob(["\uFEFF" + csvRows.join('\n')], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
});
</script>

</body>
</html>